/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : 1-Wire Interface
 */
 
//=================================================================================================
#include <Wire.h> 
//=================================================================================================
#include "DS2482.h"
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#include <HardwareSerial.h>
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialNBIOT_RX_PIN    14
#define SerialNBIOT_TX_PIN    13
#define SerialNBIOT  Serial1                                                                      // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define SIM7020E_PWRKEY_PIN   33                                                                  // ESP32-WROVER :IO33
#define SIM7020E_SLEEP_PIN    32                                                                  // ESP32-WROVER :IO32
#define SIM7020E_PWRKEY_LOW   LOW                                                                 // Start Power-ON
#define SIM7020E_PWRKEY_HIGH  HIGH                                                                // Release Signal
#define SIM7020E_SLEEP_LOW    LOW                                                                 // Pull-Up DTR to Enable Sleep
#define SIM7020E_SLEEP_HIGH   HIGH                                                                // DTR=Low(Wakeup From Sleep)
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================

//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

//=================================================================================================
DS2482 ds(0);  //channels ds2482-800 is 0 to 7, DS2482-100 is just set 0
//=================================================================================================
byte i;
byte present = 0;
byte type_s;
byte data[12];
byte addr[8];
float celsius, fahrenheit;
//=================================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialNBIOT.begin(115200, SERIAL_8N1, SerialNBIOT_RX_PIN, SerialNBIOT_TX_PIN);
  while(!SerialNBIOT);
  //===============================================================================================
  SerialRS485.begin(9600, SERIAL_8N1, SerialRS485_RX_PIN, SerialRS485_TX_PIN);
  while(!SerialRS485);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo 1-Wire Interface");
  //===============================================================================================
  
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("DS2482 I2C(0x18) Test");
  //===============================================================================================
  //===============================================================================================
  SerialDebug.println("Initial DS2482...Complete");
  SerialDebug.println();
  //===============================================================================================
  lastGetI2CSensorTime = millis();
  //===============================================================================================
}
 
void loop() 
{
  //===============================================================================================
  do
  {
    if(ds.wireSearch(addr))
    {
      present = ds.wireReset(); 
      SerialDebug.print("Found Device(");
      SerialDebug.print(present, HEX);
      //
      //===========================================================================================
      // the first ROM byte indicates which chip
      //===========================================================================================
      switch(addr[0]) 
      {
        case 0x10:
          SerialDebug.print(":(DS18S20)");  // or old DS1820
          type_s = 1;
        break;
    
        case 0x28:
          SerialDebug.print(":(DS18B20)");
          type_s = 0;
        break;
    
        case 0x22:
          SerialDebug.print(":(DS1822)");
          type_s = 0;
        break;
    
        default:
          SerialDebug.print(":Unknow)");
        return;
      } 
      //===========================================================================================
      SerialDebug.print(" ROM =");
      
      for( i = 0; i < 8; i++) 
      {
        SerialDebug.write(' ');
        if(addr[i] < 16) SerialDebug.print("0");
        SerialDebug.print(addr[i], HEX);
      }
      //=============================================================================================
      ds.wireReset();
      ds.wireSelect(addr);
      ds.wireWriteByte(0x44);  // start conversion, use ds.write(0x44,1) with parasite power on at the end
      //=============================================================================================
      delay(1000);     // maybe 750ms is enough, maybe not
      //=============================================================================================
      // we might do a ds.depower() here, but the reset will take care of it.
      //=============================================================================================
      present = ds.wireReset(); //ds.reset(); 
      ds.wireSelect(addr);
      ds.wireWriteByte(0xBE);         // Read Scratchpad
      //=============================================================================================
      //Serial.print("Data(Device:");
      //Serial.print(present, HEX);
      //Serial.print(") = ");
      SerialDebug.print(" Data :");
      for ( i = 0; i < 9; i++) 
      {           // we need 9 bytes
        SerialDebug.print(" ");
        data[i] = ds.wireReadByte();
        if(data[i] < 16) SerialDebug.print("0");
        SerialDebug.print(data[i], HEX);
      }
      //Serial.print(" CRC = ");
      SerialDebug.print(":CRC(");
      if(ds.crc8(data, 8) < 16) SerialDebug.print("0");
      SerialDebug.print(ds.crc8(data, 8), HEX);
      SerialDebug.print(")");
      //=============================================================================================

      //=============================================================================================
      // Convert the data to actual temperature
      // because the result is a 16 bit signed integer, it should
      // be stored to an "int16_t" type, which is always 16 bits
      // even when compiled on a 32 bit processor.
      //=============================================================================================
      int16_t raw = (data[1] << 8) | data[0];
      //=============================================================================================
      if(type_s) 
      {
        raw = raw << 3; // 9 bit resolution default
        if (data[7] == 0x10) 
        {
          // "count remain" gives full 12 bit resolution
          raw = (raw & 0xFFF0) + 12 - data[6];
        }
      } 
      else 
      {
        //===========================================================================================
        byte cfg = (data[4] & 0x60);
        //===========================================================================================
        // at lower res, the low bits are undefined, so let's zero them
        if (cfg == 0x00) raw = raw & ~7;                // 9 bit resolution, 93.75 ms
        else if (cfg == 0x20) raw = raw & ~3;           // 10 bit res, 187.5 ms
        else if (cfg == 0x40) raw = raw & ~1;           // 11 bit res, 375 ms
        //// default is 12 bit resolution, 750 ms conversion time
        //===========================================================================================
      }
      celsius = (float)raw / 16.0;
      fahrenheit = celsius * 1.8 + 32.0;
      SerialDebug.print("  Temperature = ");
      SerialDebug.print(celsius);
      SerialDebug.print(" Celsius, ");
      SerialDebug.print(fahrenheit);
      SerialDebug.println(" Fahrenheit");
      //=============================================================================================
    }
    else
    {
      ds.wireResetSearch();
      break;
    }
  }
  //===============================================================================================
  while(1);
  //===============================================================================================
  delay(2000);
  //===============================================================================================
}

